<?php 

class SearchCacheHandler {
    private Redis $redis;
    private int $ttl; // Time-to-live in seconds

    public function __construct(int $ttlSeconds = 900) {
        $this->redis = new Redis();
        $this->redis->connect('127.0.0.1', 6379); // or use env config
        $this->ttl = $ttlSeconds;
    }

    private function makeKey(string $context, array $params): string {
        ksort($params); // Ensure consistent order
        $base = strtolower(trim($context . '|' . json_encode($params)));
        return 'search:' . md5($base);
    }

    public function get(string $context, array $params): mixed {
        $key = $this->makeKey($context, $params);
        $cached = $this->redis->get($key);
        return $cached ? json_decode($cached, true) : null;
    }

    public function set(string $context, array $params, mixed $data): bool {
        $key = $this->makeKey($context, $params);
        return $this->redis->setex($key, $this->ttl, json_encode($data));
    }

    public function clear(string $context, array $params): bool {
        $key = $this->makeKey($context, $params);
        return $this->redis->del($key) > 0;
    }

    public function clearAllByPrefix(string $prefix = 'search:'): void {
        foreach ($this->redis->keys($prefix . '*') as $key) {
            $this->redis->del($key);
        }
    }
}